#
# The following is an example of how to retrieve data from TRACE tee components.
# The calculations performed with this data are not necessarily usefull or
# appropriate but serve only as examples of how the values used can be retrieved
# from the model.
#
#

# Pre-Defined Methods:

### 
 # Retrieves the current model.
 # In batch mode the model retrieved is either the current model
 # (the last model opened, imported, etc.) or the model specified
 # by key in the MACRO command.
###
# def getModel():

### 
 # Prints the given message to the Message Window as a Notice.
###
# def addMessage(message):
addMessage( "Beginning Python data access example for TRACE Tee using W4loopn.med")

#
# In the following example, various values of Tee #10 will be displayed
# as messages in the Message Window
#

# Find tee number 10 by using it's component number and category name.
tee = findComponent("Tees",10)
# Note: The category name is the name of the category as it appears in
#       the Navigator. Using the singular "Tee" in this case will not
#       find the component and will throw an exception.

# Print the component's string representation to the Message Window
addMessage( "Example Component: "+str(tee) )

# This is a set of lines from the Tee ASCII export routine rewritten in python. 
addMessage( "*******   type           num            id        ctitle" )
addMessage( "tee                        %d             %d          %s" \
                # the tee's component number
            % ( tee.getComponentNumber(), \
                # Arbitrary user-defined ID
                0, \
                # the toString of the component.
                tee.toString()) ) 
                # toString() generates a human-readable string representation 
                # of the component for use in messages, selections, etc.
                                   
# This example show some real data being retrieved from the tee.
# NCELLS: This value is the number of cells in the first fluid segment
#         (main tube/side tube) of a fluid such as a Tee or a Tee.
#         NCELLS2 would be retrieved with: tee.getFluidSegment(1).getCellsCount()
#
# JUN1: This is the component number of the connection representing the
#       tee's inlet junction. This is a convenience method that handles
#       the 0 junction number for an unconnected inlet.
#
# JUN2: The same as JUN1 for the outlet connection. 
#       getJun3CC() is also available for components with 2 fluid segments.(tee)
#
# EPSW: Wall surface roughness.
#       The important thing to note here is that EPSW is not a float, it is an
#       object of Length. Length is a TRACE specific extension of Real which
#       is used to allow the display of numbers in either SI or British units.
#
#       NOTE: To retrieve the float value of a Real object the getDoubleValue()
#             method must be used. Also note that the value returned is always
#             in SI units.
#
addMessage( "*       ncells         nodes          jun1          jun2          epsw" )
addMessage( "             %d             %d            %d            %d          %1.2f" \
            %( tee.getFluidSegment(0).getCellsCount(), \
               tee.getNodes(), \
               tee.getJun1CC(), tee.getJun2CC(), \
               tee.getEpsw().getDoubleValue()) ) # Note the use of getDoubleValue()

# In this example, access to the second fluid segment (index 1) is shown.
addMessage( "*       iconc2        ncell2          jun3         ipow2" )
addMessage( "             %d             %d            %d             %d" \
            %( tee.getFluidSegment(1).getIconc(), \
               tee.getFluidSegment(1).getCellsCount(), \
               tee.getJun3CC(),\
               tee.getFluidSegment(1).getIpow()) )
#
# This is an example of retrieving data directly from a tee's cells in
# order to calculate the tee's total length and volume.
#
totalLength = 0.0 # initial value of 0.0
totalVolume = 0.0 # initial value of 0.0
idx = 0           # Current cell index. 
                  # Note that all indexes are in C notation {0 to (n-1)}
ncells1 = tee.getFluidSegment(0).getCellsCount()
ncells2 = tee.getFluidSegment(1).getCellsCount()

# calculate the totals for the main tube
while idx < ncells1: 
    cell = tee.getCellAt(idx)
    # Most data values follow the naming convention found in the TRACE user's manual.
    # Accessors for these values follow the Java naming convention. 
    # ( First letter is lower case; Each successive word is capitalized. )
    totalLength += cell.getDx().getDoubleValue()
    totalVolume += cell.getVol().getDoubleValue()
    idx += 1 # next cell
    # similar code can be used to retrieve other cell values such as:
    # getAlp(), getTl(), getTv(), getP(), getConc(), getS(), etc.
    
addMessage( "Main Tube Total Length: %1.3f m" % totalLength )
addMessage( "Main Tube Total Volume: %1.3f m^3" % totalVolume )


# calculate the totals for the side tube
while idx < ncells2: 
    # offset by ncells1 to get the proper index
    cell = tee.getCellAt(ncells1 + idx) 
    totalLength += cell.getDx().getDoubleValue()
    totalVolume += cell.getVol().getDoubleValue()
    idx += 1 # next cell
    
addMessage( "Side Tube Total Length: %1.3f m" % totalLength )
addMessage( "Side Tube Total Volume: %1.3f m^3" % totalVolume )


#
# This example shows retrieving data from a tee's edges for displaying
# each edge's flow area, hydraulic diameter and GRAV.
#
idx = 0           # Current edge index. 
                  # Note that all indexes are in C notation {0 to (n-1)}
addMessage( "   %4s %14s %14s %14s" %( "edge", "flow area", "hyd diam", "grav" ) )

while idx < tee.getEdgeCount():# getEdgeCount is the total number of edges
    # findEdgeAt must be used rather than getEdgeAt to retrieve the appropriate
    # edge when edge data is being shared between fluid components.
    # Edge data that would be duplicated in the TRACE ASCII deck is shared
    # in the plugin.
    edge = tee.findEdgeAt(idx) 
    # Display the edge number, flow area, hydraulic diameter and grav term.
    # Use the toString for FA and HD in order to use the number formatting
    # features of the Real class.
    addMessage( "   %4d %14s %14s %14f" % \
                   #adjust idx to fortran notation {1-n}
                ( (idx+1), \
                  edge.getFa().toString(), \
                  edge.getHd().toString(), \
                  edge.getGrav() ) )
    idx += 1 # next edge
    # similar code can be used to retrieve other edge values such as:
    # getFric(), getFricr(), getVl(), getVv(), getHd_ht()
